package com.sap.idm.vds.connector.samplefile.impl;

import java.util.Vector;

import com.sap.idm.vds.filter.MVDFilter;
import com.sap.idm.vds.filter.MVDFilterComplex;
import com.sap.idm.vds.filter.MVDFilterSimple;
import com.sap.idm.vds.filter.MVDFilterUtilities;


public class FilterTransformer {

	/**
     * Transforms a filter from the original format given by the next syntax in BNF:
     * 
     * <filter> ::= <complex> | <simple>
	 * <simple> ::= sentence
	 * <complex> ::= (<operator>) | (<operator> <complex>)
	 * <operator> ::= ! <sentence> | & <sentences> | ‘|’ <sentences> | 
	 *					& <complex> <sentences> | ‘|’ <complex> <sentences> |
	 *					& <sentences> <complex> | ‘|’ <sentences> <complex> |
	 *					! <complex> | & <complex> | ‘|’ <complex>
	 *					& <sentences> <complex> <sentences> | 
	 *					‘|’ <sentences> <complex> <sentences>
	 * <sentences> ::= <sentence> | <sentence> <sentences>
	 * <sentence> ::= (<string> <op> <string>)
	 * <string> ::= 0|1|…|9|a|…|z|A|…|Z|*|…
	 * <op> ::= >= | <= | =
     * 
     * 
     * to the goal format given by the syntax:
     * 
     * 
     * <filter> ::= <simple> | <complex>
     * <simple> ::= (<string> <op> <string>
	 * <complex> ::= (NOT <filter>) | (<filter> <and>) | (<filter> <or>)
	 * <and> ::= AND <filter> | AND <filter> <and>
	 * <or> ::= OR <filter> | OR <filter> <or>
	 * <string> ::= 0|1|…|9|a|…|z|A|…|Z|*|…
	 * <op> ::= >= | <= | =
     * 
     * 
     * @param filter -> Filter
     * @return -> The goal filter
     */
    private String toMyFormatFilter (String filter) {
    	
    	MVDFilter mFilter = MVDFilterUtilities.constructFilter(filter);
    	return this.toMyFormatFilter(mFilter);
    }
    
    
    /**
     * Calls to the right method depending on if the filter is complex or simple
     * @param aMVDFilter -> Filter
     * @return -> The goal filter
     */
	private String toMyFormatFilter(MVDFilter aMVDFilter) {
		/* Checks if the filter is complex or simple and calls to the correct method */
		if(aMVDFilter instanceof MVDFilterSimple)
			return this.toMyFormatFilter((MVDFilterSimple)aMVDFilter);

		return this.toMyFormatFilter((MVDFilterComplex)aMVDFilter);
	}

	
	/**
	 * Transforms a simple original filter to a simple goal filter
	 * @param aMVDFilterSimple -> The simple filter
	 * @return -> The goal filter for the corresponding simple filter
	 */
	private String toMyFormatFilter(MVDFilterSimple aMVDFilter) {
		
		String opstr = "";
		String value = null;  
		/* Gets the attribute name */
		String mAttrname=aMVDFilter.getAttrName();
		/* Gets the attribute value */
		String mAttrvalue=aMVDFilter.getAttrValue();
		/* Gets the operation type */
		int mOptype=aMVDFilter.getOpType();	     	     
		
		/* An null attribute name will be considered as that any attribute name is good */
		if (mAttrname==null) mAttrname="*";
		
		/* Checks if the all the attributes cleaned from the filter */
		if (mAttrname.equals(MVDFilter.MVD_CLEANED)) return null;
		/* Checks if the all the attributes ignored */
		if (mAttrname.equals(MVDFilter.MVD_IGNORED)) return "(*=*)";
			    
		/* For null attributes we assume that that means that any value would be good */
		if (mAttrvalue == null) value = "*";
		else value = mAttrvalue;
		
		/* If we receives a invalid operator type it will be considered as EQUAL type */
		if ((mOptype == MVDFilter.OP_EQUAL) ||  (mOptype == MVDFilter.FILTER_OP_INVALID)) {
		    opstr = "=";
		}
		/* Operator greater than */
		else if (mOptype == MVDFilter.OP_GT) {
		    opstr = ">";
		}
		/* Operator present. It will considered as EQUAL operator */
		else if (mOptype == MVDFilter.OP_PRESENT) {
		    opstr = "=";
		}
		/* Operator less than */
		else if (mOptype == MVDFilter.OP_LT) {
		    opstr = "<";
		}
		/* The substring operator will be considered as EQUAL operator */
		else if (mOptype == MVDFilter.OP_SUBSTRING) {
		    opstr = "=";
		}
		/* The filter with the expected syntax is constructed */
		return "(" + mAttrname + opstr + value + ")";
	}

	
	/**
	 * Transforms a complex original filter to a complex goal filter
	 * @param aMVDFilterComplex -> The complex filter
	 * @return -> The goal filter for the corresponding complex filter
	 */
	private String toMyFormatFilter(MVDFilterComplex aMVDFilter) {
			 	
		String tmp = "";
		/* Gets the operator type of this complex filter */
		int mFilterOpType=aMVDFilter.getFilterOpType();
		/* Gets the set of associated filters to this operator */
	    Vector mFilters=aMVDFilter.getFilters();
		/* Takes the first associated filter */
        MVDFilter mOne = (MVDFilter) mFilters.elementAt(0);
        if (mOne != null) {
        	/* For NOT operator there will be only an associated filter */
            if (mFilterOpType == MVDFilter.FILTER_OP_NOT) {
            	if(mOne instanceof MVDFilterComplex) {
            		String aux = toMyFormatFilter((MVDFilterComplex)mOne);
            		if (aux==null) return null;
            		tmp="(NOT "+aux+")";
            	}
            	else {
            		String aux = toMyFormatFilter((MVDFilterSimple)mOne);
            		if (aux==null) return null;
            		tmp="(NOT "+aux+")";
            	}
            	return tmp;
            }
            /* For AND and OR operations at least two associated filters are necessary */
            else if (mFilters.size() > 1) {
                if (mFilterOpType == MVDFilter.FILTER_OP_AND) {
                	/* For AND operator we will check the complete set of associated filters and we will
                	 * join them with the string "AND" */
                	tmp="(";
                    for (int x = 0; x <  mFilters.size(); x++) {
                    	/* Between every two consecutive filters a "AND" will be inserted */
                    	if (x>0) tmp+=" AND ";
                    	if(mFilters.elementAt(x) instanceof MVDFilterComplex) {
                    		String aux = toMyFormatFilter((MVDFilterComplex)mFilters.elementAt(x));
                    		if (aux==null) return null;
                    		tmp += aux;
                    	} else {
                    		String aux = toMyFormatFilter((MVDFilterSimple)mFilters.elementAt(x));
                    		if (aux==null) return null;
                    		tmp += aux;
                        }
                    }
                   return tmp+")";
                }
                else if (mFilterOpType == MVDFilter.FILTER_OP_OR) {
                	/* For OR operator we will check the complete set of associated filters and we will
                	 * join them with the string "OR" */
                	tmp="(";
                    for (int x = 0; x <  mFilters.size(); x++) {
                    	/* Between every two consecutive filters a "OR" will be inserted */
                    	if (x>0) tmp+=" OR ";
                    	if(mFilters.elementAt(x) instanceof MVDFilterComplex) {
                    		String aux = toMyFormatFilter((MVDFilterComplex) mFilters.elementAt(x));
                    		if (aux==null) return null;
                    		tmp += aux;
                    	} else {
                    		String aux = toMyFormatFilter((MVDFilterSimple) mFilters.elementAt(x));
                    		if (aux==null) return null;
                    		tmp += aux;
                    	}
                    }
                    return tmp+")";
                }
            }
            else {
                // --- there should be at least two of partial filters
            	return null;
            }
        }
        return null;
	}
	
	
	public static void main (String []args) {
		
		FilterTransformer ft = new FilterTransformer();
		
		String [] filters = {"(a=1)",
							 "(&(a=1)(b=2)(|(c=3)(d=4)(!(e=5))))",
							 "(!(&(a=1)(b=2)))",
							 "(|(&(a=1)(b=2))(c=3))",
							 "(&(a=1))",
							 "(|((a=1)(b=2)))",
							 "()",
							 "(|)",
							 "(|()())",
							 "(|(&(|(!(a=1))(b=2))(c=3))(d=4))",
							 "(|(&(|(!(a=1))(b=2))(c=3)))"
							};
		
		for (int i=0; i<filters.length; i++) {
			String res = ft.toMyFormatFilter(filters[i]);
			if (res==null) res="null";
			System.out.println(filters[i]+" -> "+res);
		}
	}
}
